// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

#nullable enable

using System;
using FluentAssertions;
using NuGet.Protocol.Model;
using NuGet.Versioning;
using Xunit;

namespace NuGet.Protocol.Tests.Model
{
    public class VulnerabilityInfoTests
    {
        [Fact]
        void Equals_TwoInstancesWithSameValues_AreEqual()
        {
            // Arrange
            Uri cveUrl = new("https://contoso.test/cve/1");
            VersionRange versionRange = new(minVersion: new NuGetVersion(1, 0, 0), includeMinVersion: true, maxVersion: new NuGetVersion(2, 0, 0), includeMaxVersion: false);
            PackageVulnerabilityInfo v1 = new(cveUrl, PackageVulnerabilitySeverity.Moderate, versionRange);
            PackageVulnerabilityInfo v2 = new(cveUrl, PackageVulnerabilitySeverity.Moderate, versionRange);

            // Act & Assert
            v1.Equals(v2).Should().BeTrue();
            v1.Equals((object)v2).Should().BeTrue();
            v1.GetHashCode().Should().Be(v2.GetHashCode());
        }

        [Fact]
        void Equals_TwoInstancesWithDifferentURLs_AreNotEqual()
        {
            // Arrange
            Uri cveUrl1 = new("https://contoso.test/cve/1");
            Uri cveUrl2 = new("https://contoso.test/CVE/1");
            VersionRange versionRange = new(minVersion: new NuGetVersion(1, 0, 0), includeMinVersion: true, maxVersion: new NuGetVersion(2, 0, 0), includeMaxVersion: false);
            PackageVulnerabilityInfo v1 = new(cveUrl1, PackageVulnerabilitySeverity.Moderate, versionRange);
            PackageVulnerabilityInfo v2 = new(cveUrl2, PackageVulnerabilitySeverity.Moderate, versionRange);

            // Act & Assert
            v1.Equals(v2).Should().BeFalse();
            v1.Equals((object)v2).Should().BeFalse();
            v1.GetHashCode().Should().NotBe(v2.GetHashCode());
        }

        [Fact]
        void Equals_TwoInstancesWithDifferentSeverity_AreNotEqual()
        {
            // Arrange
            Uri cveUrl = new("https://contoso.test/cve/1");
            VersionRange versionRange = new(minVersion: new NuGetVersion(1, 0, 0), includeMinVersion: true, maxVersion: new NuGetVersion(2, 0, 0), includeMaxVersion: false);
            PackageVulnerabilityInfo v1 = new(cveUrl, PackageVulnerabilitySeverity.Moderate, versionRange);
            PackageVulnerabilityInfo v2 = new(cveUrl, PackageVulnerabilitySeverity.High, versionRange);

            // Act & Assert
            v1.Equals(v2).Should().BeFalse();
            v1.Equals((object)v2).Should().BeFalse();
            v1.GetHashCode().Should().NotBe(v2.GetHashCode());
        }

        [Fact]
        void Equals_TwoInstancesWithDifferentRanges_AreNotEqual()
        {
            // Arrange
            Uri cveUrl = new("https://contoso.test/cve/1");
            VersionRange versionRange1 = new(minVersion: new NuGetVersion(1, 0, 0), includeMinVersion: true, maxVersion: new NuGetVersion(2, 0, 0), includeMaxVersion: false);
            VersionRange versionRange2 = new(minVersion: new NuGetVersion(1, 0, 1), includeMinVersion: true, maxVersion: new NuGetVersion(2, 0, 0), includeMaxVersion: false);
            PackageVulnerabilityInfo v1 = new(cveUrl, PackageVulnerabilitySeverity.Moderate, versionRange1);
            PackageVulnerabilityInfo v2 = new(cveUrl, PackageVulnerabilitySeverity.Moderate, versionRange2);

            // Act & Assert
            v1.Equals(v2).Should().BeFalse();
            v1.Equals((object)v2).Should().BeFalse();
            v1.GetHashCode().Should().NotBe(v2.GetHashCode());
        }
    }
}
