/*****************************************************************************
 *                                                                           *
 *          UNURAN -- Universal Non-Uniform Random number generator          *
 *                                                                           *
 *****************************************************************************
 *                                                                           *
 *   FILE: urng_fvoid.h                                                      *
 *                                                                           *
 *   PURPOSE:                                                                *
 *     Function prototypes for meta uniform random number generator to       *
 *     combine point set generator with random shifts.                       *
 *                                                                           *
 *   USAGE:                                                                  *
 *         only included in unuran.h                                         *
 *                                                                           *
 *****************************************************************************
 *                                                                           *
 *   Copyright (c) 2000-2006 Wolfgang Hoermann and Josef Leydold             *
 *   Department of Statistics and Mathematics, WU Wien, Austria              *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Public License as published by    *
 *   the Free Software Foundation; either version 2 of the License, or       *
 *   (at your option) any later version.                                     *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU General Public License for more details.                            *
 *                                                                           *
 *   You should have received a copy of the GNU General Public License       *
 *   along with this program; if not, write to the                           *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place, Suite 330, Boston, MA 02111-1307, USA                  *
 *                                                                           *
 *****************************************************************************/

/*---------------------------------------------------------------------------*/
#ifndef URNG_RANDOMSHIFT_H_SEEN
#define URNG_RANDOMSHIFT_H_SEEN
/*---------------------------------------------------------------------------*/

/* 
   =NODE  URNG-RANDOMSHIFT  Combine point set generator with random shifts

   =UP URNG [90]

   =DESCRIPTION
      Generators of type RANDOMSHIFT combine a point set generator with
      generators to apply random shifts as proposed in
      @unurbibref{CPa76}:

      @enumerate
      @item
      Sample and store a random vector S.
      @item
      Run a QMC simulation where S is added to each point of the
      generated quasi-random point (mod 1).
      @item
      Repeat steps 1 and 2.
      @end enumerate

   =HOWTOUSE
      Create a URNG object for a point set generator and a URNG object
      for a generator to create shift vectors at random.
      The meta URNG object can then be created using
      unur_urng_randomshift_new().
      Notice that only pointers to the two underlying URNG generator
      objects are copied into the newly created meta generator. Thus 
      manipulating the meta URNG also changes the underlying URNGs
      and vice versa.

      The following routines are supported for URNG objects of
      type RANDOMSHIFT:

      @itemize @minus
      @item unur_urng_sample()
      @item unur_urng_sample_array()
      @item unur_urng_reset() 
      @item unur_urng_sync() 
      @item unur_urng_randomshift_nextshift()
      @item unur_urng_free()
      @end itemize

      unur_urng_sync() is used to jump to the first coordinate of
      the next point generated by the generator. 

      unur_urng_randomshift_nextshift() allows to replace the shift
      vector by another randomly chosen shift vector.

      @emph{Important:} unur_urng_sync() is only available if it is
      if it is implemented for the underlying point set generator.

      @emph{Important:} unur_urng_reset() is only available if it is
      available for both underlying generators. 

   =END

*/

/*---------------------------------------------------------------------------*/

/* =ROUTINES */

/*---------------------------------------------------------------------------*/

UNUR_URNG *unur_urng_randomshift_new( UNUR_URNG *qrng, UNUR_URNG *srng, int dim );
/* 
   Make object for URNG with randomly shifted point sets.
   @var{qrng} is a generated that generates point sets of dimension @var{dim}.
   @var{srng} is a generated that generates random numbers or vectors.

   @emph{Notice:} Only pointers to the respective objects @var{qrng}
   and @var{srng} are copied into the created meta generator. Thus
   manipulating the meta URNG also changes the underlying URNGs
   and vice versa.
*/

int unur_urng_randomshift_nextshift( UNUR_URNG *urng );
/* 
   Get the next (randomly chosen) vector for shifting the points set, and the 
   underlying point generator @var{qrng} is reset.
*/

/*
   =END
*/

/*---------------------------------------------------------------------------*/
#endif  /* URNG_RANDOMSHIFT_H_SEEN */
/*---------------------------------------------------------------------------*/
