/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/

#include "creategroupdialog.h"
#include "userinfo.h"
#include "closebutton.h"
#include "changeusergroup.h"

#define GROUPNAME_LENGTH 32

extern void qt_blurImage(QImage &blurImage, qreal radius, bool quality, int transposed);

CreateGroupDialog::CreateGroupDialog(QWidget *parent) :
    QDialog(parent),
    cgDialog(new changeUserGroup),
    _nameHasModified(false),
    _idHasModified(false)
{
    vlayout = new QVBoxLayout(this);
    vlayout->setContentsMargins(22, 0, 25, 0);
    initUI();
    signalsBind();
}

CreateGroupDialog::~CreateGroupDialog()
{
    delete cgDialog;
    cgDialog = nullptr;
}

void CreateGroupDialog::initUI()
{
    this->setWindowTitle(tr("Add user group"));
    this->setFixedSize(480, 214);
    setAttribute(Qt::WA_DeleteOnClose);

    mGroupNameLabel = new FixLabel(tr("Name"), this);
    mGroupNameLabel->setFixedSize(62, 36);
    mGroupNameEdit = new QLineEdit(this);
    mGroupNameEdit->installEventFilter(this);
    mGroupNameEdit->setFixedSize(370, 36);

    mGroupNameHLayout = new QHBoxLayout(this);
    mGroupNameHLayout->setContentsMargins(0,0,0,0);
    mGroupNameHLayout->setSpacing(2);
    mGroupNameHLayout->addWidget(mGroupNameLabel);
    mGroupNameHLayout->addWidget(mGroupNameEdit);

    mGroupNameTipLabel = new FixLabel();
    mGroupNameTipLabel->setFixedWidth(370);
    QFont ft;
    ft.setPixelSize(14);
    mGroupNameTipLabel->setFont(ft);
    mGroupNameTipLabel->setFixedSize(QSize(370, 24));
    mGroupNameTipLabel->setStyleSheet("color:red;");
    mGroupNameTipHLayout = new QHBoxLayout();
    mGroupNameTipHLayout->setContentsMargins(0,0,0,0);
    mGroupNameTipHLayout->addStretch();
    mGroupNameTipHLayout->addWidget(mGroupNameTipLabel);

    mGroupIdLabel = new FixLabel(tr("Id"), this);
    mGroupIdLabel->setFixedSize(62, 36);
    mGroupIdEdit = new QLineEdit(this);
    mGroupIdEdit->setFixedSize(370, 36);

    mGroupIdHLayout = new QHBoxLayout(this);
    mGroupIdHLayout->setContentsMargins(0,0,0,0);
    mGroupIdHLayout->setSpacing(2);
    mGroupIdHLayout->addWidget(mGroupIdLabel);
    mGroupIdHLayout->addWidget(mGroupIdEdit);

    mCancelBtn = new QPushButton(tr("Cancel"), this);
    mCancelBtn->setFixedSize(96, 36);
    mCancelBtn->setFocusPolicy(Qt::FocusPolicy::NoFocus);
    mConfirmBtn = new QPushButton(tr("Confirm"), this);
    mConfirmBtn->setMinimumSize(96, 36);

    mConfirmHLaout = new QHBoxLayout(this);
    mConfirmHLaout->setContentsMargins(0,0,0,0);
    mConfirmHLaout->addStretch();
    mConfirmHLaout->addWidget(mCancelBtn);
    mConfirmHLaout->addWidget(mConfirmBtn);

    vlayout->addStretch();
    vlayout->addSpacing(20);
    vlayout->addLayout(mGroupNameHLayout);
    vlayout->addLayout(mGroupNameTipHLayout);
    vlayout->addLayout(mGroupIdHLayout);
    vlayout->addSpacing(40);
    vlayout->addLayout(mConfirmHLaout);
    vlayout->addSpacing(24);
    vlayout->addStretch();

    setLayout(vlayout);

    // 设置确认按钮
    refreshCertainBtnStatus();
    // 限制组名输入规则
    limitInput();
}


void CreateGroupDialog::limitInput()
{
    QIntValidator *intValidator = new QIntValidator;
    // QRegExp rx("^[a-zA-z]+$");// 首字符为字母
    QRegExp rx("^[a-zA-Z][a-zA-Z0-9_-]*${32}");
    QRegExpValidator *regValidator = new QRegExpValidator(rx);
    // intValidator->setRange(0, 65535);
    intValidator->setBottom(0);
    // 整形输入限制
    mGroupIdEdit->setValidator(intValidator);
    // 字母输入限制
    mGroupNameEdit->setValidator(regValidator);
    // 字符长度限制
    // ui->lineEdit_name->setMaxLength(4);
}

void CreateGroupDialog::refreshCertainBtnStatus()
{
    if (mGroupNameEdit->text().isEmpty()
        || mGroupIdEdit->text().isEmpty())
        mConfirmBtn->setEnabled(false);
    else
        mConfirmBtn->setEnabled(_nameHasModified || _idHasModified);
}

QPushButton *CreateGroupDialog::certainBtnComponent()
{
    return mConfirmBtn;
}

QLineEdit *CreateGroupDialog::lineNameComponent()
{
    return mGroupNameEdit;
}

QLineEdit *CreateGroupDialog::lineIdComponent()
{
    return mGroupIdEdit;
}

void CreateGroupDialog::signalsBind()
{
    connect(mCancelBtn, &QPushButton::clicked, [=](){
        close();
    });
    connect(mGroupNameEdit, &QLineEdit::textChanged, [=](QString txt){
        Q_UNUSED(txt);
        refreshCertainBtnStatus();
    });
    connect(mGroupIdEdit, &QLineEdit::textChanged, [=](QString txt){
        Q_UNUSED(txt);
        refreshCertainBtnStatus();
    });
    connect(mGroupIdEdit, &QLineEdit::textEdited, [=](){
        for (int j=0; j < cgDialog->groupList->size(); j++) {
            if (mGroupIdEdit->text() == cgDialog->groupList->at(j)->groupid) {
                _idHasModified = false;
                break;
            } else {
                if (j == cgDialog->groupList->size() - 1) {
                    _idHasModified = true;
                }
            }
        }
    });
    connect(mGroupNameEdit, &QLineEdit::textEdited, [=](QString txt){
        for (int j=0; j < cgDialog->groupList->size(); j++) {
            if (mGroupNameEdit->text() == cgDialog->groupList->at(j)->groupname) {
                _nameHasModified = false;
                break;
            } else {
                if (j == cgDialog->groupList->size() - 1) {
                    _nameHasModified = true;
                }
            }
        }
        if (mGroupNameEdit->text().length() > GROUPNAME_LENGTH) {
            mGroupNameEdit->setText(oldGroupName);
        } else {
            oldGroupName = txt;
        }
        if(mGroupNameEdit->text().isEmpty()) {
            mGroupNameTip = tr("GroupName's length must be between 1 and %1 characters!").arg(GROUPNAME_LENGTH);
        } else {
            mGroupNameTip = "";
        }

        mGroupNameTipLabel->setText(mGroupNameTip);
    });
}

bool CreateGroupDialog::eventFilter(QObject *watched, QEvent *event){

    if (event->type() == QEvent::FocusOut) {
        if (watched == mGroupNameEdit) {
            if (mGroupNameEdit->text().isEmpty()) {
                mGroupNameTip = tr("GroupName's length must be between 1 and %1 characters!").arg(GROUPNAME_LENGTH);
                mGroupNameTipLabel->setText(mGroupNameTip);
            }
        }
    }
    return QObject::eventFilter(watched, event);
}
